/*
  ESP32 + HX711 + OLED SSD1306 (128x64)
  Prosta waga z szybką reakcją i lekkim wygładzaniem.

  Sterowanie z Serial (monitor portu szeregowego, 115200):
    t  -> tare (zerowanie)
    c  -> kalibracja (podaj masę wzorca)
    ?  -> pomoc

  Podłączenie (ESP32 DevKit):
    HX711 DOUT -> GPIO19
    HX711 SCK  -> GPIO18
    OLED  SDA  -> GPIO21
    OLED  SCL  -> GPIO22
    VCC 3.3 V, GND wspólna
*/

#include <Arduino.h>
#include <Wire.h>

#include <HX711.h>
#include <Adafruit_GFX.h>
#include <Adafruit_SSD1306.h>
#include <Preferences.h>

// --- Piny
constexpr int PIN_HX_DOUT = 19;
constexpr int PIN_HX_SCK  = 18;
constexpr int PIN_SDA     = 21;
constexpr int PIN_SCL     = 22;

// --- OLED
#define SCREEN_WIDTH 128
#define SCREEN_HEIGHT 64
Adafruit_SSD1306 display(SCREEN_WIDTH, SCREEN_HEIGHT, &Wire, -1);

// --- HX711
HX711 scale;

// --- Pamięć na współczynnik kalibracji
Preferences prefs;
const char* PREF_NS    = "scale";
const char* PREF_CAL_K = "cal";

// --- Odświeżanie OLED
unsigned long lastOled = 0;
const unsigned long OLED_PERIOD_MS = 80;  // ~12.5 Hz

// --- Stabilizacja „bez blokowania” (szybko, ale spokojnie)
const float EMA_ALPHA       = 0.28f;  // lekkie wygładzenie (0.25–0.35)
const float DISPLAY_STEP_G  = 0.2f;   // zaokrąglanie wyświetlanej wartości
const float DISPLAY_HYST_G  = 0.05f;  // histereza (trzymanie wyświetlanej wartości)

const int   MED_N = 5;                // mediana z 5 ostatnich próbek
float       medBuf[MED_N] = {0};
int         medIdx = 0;
int         medCount = 0;

bool  emaInited = false;
float gEMA      = 0.0f;
float lastShown = 0.0f;

float calFactor = 0.0f;
bool  hasCal    = false;

// --- Niewielki helper do rysowania
void drawCenteredText(const String& s, int y, int size=2) {
  int16_t x1, y1; uint16_t w, h;
  display.setTextSize(size);
  display.getTextBounds(s, 0, y, &x1, &y1, &w, &h);
  display.setCursor((SCREEN_WIDTH - w) / 2, y);
  display.print(s);
}

void showWeight(float grams) {
  display.clearDisplay();
  display.setTextColor(SSD1306_WHITE);
  drawCenteredText("WAGA", 0, 1);

  display.setTextSize(3);
  display.setCursor(0, 22);
  char buf[16];
  dtostrf(grams, 0, 1, buf);   // 1 miejsce po przecinku
  display.print(buf);
  display.setTextSize(2);
  display.print(" g");

  display.setTextSize(1);
  display.setCursor(0, 56);
  display.print(hasCal ? "CAL OK" : "BRAK CAL");
  display.setCursor(72, 56);
  display.print("t=tare c=cal");
  display.display();
}

void showMsg(const __FlashStringHelper* msg1, const __FlashStringHelper* msg2 = F("")) {
  display.clearDisplay();
  display.setTextColor(SSD1306_WHITE);
  drawCenteredText(String(msg1), 16, 2);
  drawCenteredText(String(msg2), 40, 1);
  display.display();
}

// --- Mediana z bufora (N<=5) – gasi pojedyncze „strzały” szumu
static float medianFromBuffer(const float* buf, int n) {
  float tmp[MED_N];
  for (int i=0; i<n; ++i) tmp[i] = buf[i];
  for (int i=1; i<n; ++i) {                 // insertion sort
    float key = tmp[i];
    int j = i - 1;
    while (j >= 0 && tmp[j] > key) { tmp[j+1] = tmp[j]; --j; }
    tmp[j+1] = key;
  }
  if (n == 0) return 0.0f;
  return (n % 2) ? tmp[n/2] : 0.5f * (tmp[n/2 - 1] + tmp[n/2]);
}

// --- Kalibracja: najpierw tare, potem kładziesz wzorzec i potwierdzasz Enterem, a na końcu wpisujesz jego masę
void runCalibration() {
  showMsg(F("KALIBRACJA"), F("Patrz Serial"));
  Serial.println(F("\n=== KALIBRACJA ==="));
  Serial.println(F("1) Zdejmij wszystko. Zerowanie..."));
  scale.tare(30);
  Serial.println(F("OK."));

  while (Serial.available()) Serial.read();
  Serial.println(F("2) Poloz wzorzec (np. 1000 g). Gdy lezy i sie nie rusza, wcisnij Enter (lub 'q' aby przerwac)."));
  Serial.print(F("> "));

  while (true) {
    if (Serial.available()) {
      String line = Serial.readStringUntil('\n');
      line.trim();
      if (line.equalsIgnoreCase("q")) {
        Serial.println(F("Kalibracja przerwana."));
        showMsg(F("CAL PRZERW."));
        delay(800);
        return;
      }
      break; // Enter => zaczynamy pomiar
    }
    delay(10);
  }

  Serial.println(F("Mierze..."));
  long delta = scale.get_value(10);   // krótkie uśrednienie surowych countów
  Serial.print(F("Zebrane odczyty: "));
  Serial.println(delta);

  Serial.println(F("3) Wpisz mase wzorca w gramach i Enter (lub 'q' aby przerwac):"));
  Serial.print(F("> "));
  float known_g = NAN;
  while (true) {
    if (Serial.available()) {
      String line = Serial.readStringUntil('\n');
      line.trim();
      if (line.equalsIgnoreCase("q")) {
        Serial.println(F("Kalibracja przerwana."));
        showMsg(F("CAL PRZERW."));
        delay(800);
        return;
      }
      known_g = line.toFloat();
      if (isfinite(known_g) && known_g > 0.0f) break;
      Serial.println(F("Podaj dodatnia liczbe (g):"));
      Serial.print(F("> "));
    }
    delay(10);
  }

  float newCal = delta / known_g;     // count/gram
  if (!isfinite(newCal) || newCal <= 0.0f) {
    Serial.println(F("Blad kalibracji (sprawdz wzorzec i polaczenia)."));
    showMsg(F("CAL BLAD"), F("Sprawdz wzorzec"));
    delay(1000);
    return;
  }

  scale.set_scale(newCal);
  calFactor = newCal;
  hasCal = true;
  prefs.putFloat(PREF_CAL_K, calFactor);

  Serial.print(F("CAL_FACTOR = "));
  Serial.println(calFactor, 6);
  Serial.println(F("Zapisano w NVS."));
  showMsg(F("CAL OK"), F("Zapisano"));
  delay(700);

  // restart lekkiego wygładzania po skoku kalibracji
  emaInited = false;
  medIdx = 0; medCount = 0; lastShown = 0.0f;
}

// --- Pomoc
void printHelp() {
  Serial.println(F("\nKomendy: t=tare, c=kalibracja, ?=pomoc"));
}

void setup() {
  Serial.begin(115200);
  delay(120);

  Wire.begin(PIN_SDA, PIN_SCL);
  Wire.setClock(100000);

  display.begin(SSD1306_SWITCHCAPVCC, 0x3C);
  showMsg(F("ESP32 WAGA"), F("HX711 + OLED"));

  scale.begin(PIN_HX_DOUT, PIN_HX_SCK);
  delay(200);
  scale.tare(10);

  // Wczytanie kalibracji z pamięci
  prefs.begin(PREF_NS, false);
  calFactor = prefs.getFloat(PREF_CAL_K, 0.0f);
  if (calFactor > 0.0f && isfinite(calFactor)) {
    scale.set_scale(calFactor);
    hasCal = true;
  } else {
    scale.set_scale();  // tryb surowy do czasu kalibracji
    hasCal = false;
  }

  printHelp();
}

void loop() {
  // Komendy z Serial: pojedyncze znaki
  if (Serial.available()) {
    char c = (char)Serial.read();
    if (c == 't') {
      scale.tare(10);
      Serial.println(F("Tare OK."));
      showMsg(F("TARE OK"));
      emaInited = false; medIdx = 0; medCount = 0; lastShown = 0.0f;
      delay(150);
    } else if (c == 'c') {
      runCalibration();
    } else if (c == '?') {
      printHelp();
    }
  }

  // Jedna szybka próbka na pętlę (maks. responsywność)
  float gRaw = scale.get_units(1);
  if (!isfinite(gRaw)) gRaw = 0.0f;

  // Mediana 5 próbek – tanie tłumienie skoków bez opóźnień
  medBuf[medIdx++] = gRaw;
  if (medIdx >= MED_N) medIdx = 0;
  if (medCount < MED_N) medCount++;
  float gMed = medianFromBuffer(medBuf, medCount);

  // Lekkie EMA – wygładza resztę szumu
  if (!emaInited) { gEMA = gMed; emaInited = true; }
  else            { gEMA = EMA_ALPHA * gMed + (1.0f - EMA_ALPHA) * gEMA; }

  // Zaokrąglanie + histereza wyświetlania (uspokaja cyferki)
  float gQuant = roundf(gEMA / DISPLAY_STEP_G) * DISPLAY_STEP_G;
  float gShow  = (fabsf(gQuant - lastShown) < DISPLAY_HYST_G) ? lastShown : (lastShown = gQuant);

  // Wyświetlaj co ~80 ms (mniej migotania, mniejsze obciążenie I2C)
  unsigned long now = millis();
  if (now - lastOled >= OLED_PERIOD_MS) {
    lastOled = now;
    showWeight(gShow);
  }

  delay(5);
}
